/*jshint esversion: 6 */
/* global define, console */

define(["src/utils", "src/build/DisplayContainerStorage", "src/math/Mat3", "src/build/Container", "src/build/Puppet", "src/build/SdkLayer",
		"src/math/Vec2", "lodash", "src/build/StageId", "src/build/Handle", 
		"lib/dev", "lib/dev/disableWhen",
		"src/build/ZoneProfiler", "lib/tasks/treeOfMatrix", "lib/tasks/dofs",
       "lib/dev/config"],
function (utils, DisplayContainerStorage, mat3, Container, Puppet, SdkLayer,
		vec2, lodash, StageId, Handle, 
		dev, disableWhen,
		ZoneProfiler, treeOfMatrix, dofs, config) {
	"use strict";

	var kOriginLeaf = Handle.kOriginLeaf,
		kFullAuto = { translation : true, linear : true };

    /*
	function hasFreeJoint (layer) {
		const mode = layer.motionMode;
		return mode.translation === true && mode.linear === true;
	}
    */

    function attachmentRestrictsLayerMotion (layer) {
	 	// if this layer attaches to another
	 	return layer.isAttached;
         
        /* TODO: (?) change to
                // if this layer attaches to another *with* some restriction
                // then set the desired joint dofs between the two
                return layer.isAttached && !hasFreeJoint(layer);
          
                so that when set to Free we don't create a leaf-drop handle thus making the mesh more complex than we need; however, we tried that and it stops Dangle from working (perhaps because it's
                currently set to treat Free as Hinge, to avoid things falling just because they are dangling, and
                when it tries to pin the hinge, it pins all children of the hinge, stopping them from dangling.
                Unclear, but for now (July 2017, for 1.1 release) we're keeping this the same as Beta 6 & earlier.
                With our new default of Weld instead of Free, the Dangle-falling issue may fade away. Or with
                triggerable throwing tied to motion-mode being dynamic, this optimization wouldn't make sense
                anymore anyway -- we'll always need the origin to have a handle on the mesh just in case.
                
                note that a change here requires a change to sWarpContainerHashVersion too
        */
	 }

	function gatherLeafInitMatrix (layer) {
		var tree = layer.getHandleTreeArray(),
			aMatLayer_Handle = tree.getAccumulatedTreeAtRest("layer"),
			aLeafRef = tree.getLeafRefArray();

		return lodash.at(aMatLayer_Handle, aLeafRef);
	}

	function makeJointHandle (layer) {
		var root = layer.getHandleTreeRoot(),
			matEye = mat3.identity(),
			joint = new Handle({
				name : kOriginLeaf, 
				puppet : layer.getPuppet(),
				locations : root.transformLocations(matEye),
				anchor : root.transformAnchor(matEye)
			}),
			// by default joint handle is immediate child of the origin...
			parent = layer.getHandleTreeRoot(),
			// ... unless the origin coincides with another handle			
			warper = layer.getWarperLayer(),
			aLeaf = warper.gatherHandleLeafArray(),
			aMat_Leaf = gatherLeafInitMatrix(warper),
			pt_Origin = warper.getHandleMatrixAtRest(parent, mat3()).getTranslation(vec2()),
			parentId = lodash.findIndex(aMat_Leaf, function (mat_Leaf) {
				return pt_Origin.equalsApproximately(mat_Leaf.getTranslation());
			});

		// use collocated handle as the parent instead of the origin
		if (parentId > -1) parent = aLeaf[parentId];

		// note: added as the first child so that it dominates other children when they are located at the same location
		// see removal of invalid handles in PuppetWarp/Beaker.cpp:Solver::AddPointHandles()
		parent.addChild(joint, 0);		

		return joint;
	}

	function setupJoint (layer) {
		// look for already existing joint (i.e. after cloning) so that we don't create another one
		var aLeaf = layer.gatherHandleLeafArray();
		var joint = lodash.find(aLeaf, function (leaf) {
			if (!leaf.isOriginLeaf()) return false;
			return leaf.getPuppet() === layer.getPuppet();
		});

		if (layer.getWarpWithParent()) {
			var layerAttachedToOrigin = layer.getPuppet().privateIsIndependentLayerAttachedToOrigin();
			if (layerAttachedToOrigin) {
				// a sublayer attaches to the origin of this layer,
				// so create a leaf joint to ensure that sublayer moves with the warp
				if (! joint) joint = makeJointHandle(layer); 	// reuse or create new one
				joint.privateAuto(kFullAuto);
			} else {
				// don't drop a leaf, and remove if there
				if (joint) {
					var parent = joint.getParent();
					utils.assert(parent, "setupJoint(): missing parent.");
					parent.removeChild(joint);
					joint = null;
				}
			}
		} else {
			// restrict desired joint dofs when necessary
			if (attachmentRestrictsLayerMotion(layer)) {
				if (! joint) joint = makeJointHandle(layer);	// either reuse or create new one
				joint.privateAuto(layer.motionMode);
			}
		}

		// invalidate handle tree to force rebuild after above changes
		layer.handleTreeArray = null;

		return joint;
	}

	function matRemoveShear (mat, result0) {
		var position = [], scale = [], aShear = [], aRotation = [];
		mat3.decomposeAffine(mat, position, scale, aShear, aRotation);

		// FIXME: DRY this copy from affine which i used to avoid computing the angle
		// but then also adapted for shearX = 0
		var c = aRotation[0],
			s = aRotation[1],
			px = position[0], py = position[1],
			scaleX = scale[0], scaleY = scale[1];

		var result = result0 || mat3();
		
		result[0] = c * scaleX;
		result[1] = s * scaleX;
		result[2] = 0;
		result[3] = - s * scaleY;
		result[4] = c * scaleY;
		result[5] = 0;
		result[6] = px;
		result[7] = py;
		result[8] = 1;
	}

	function matRemoveRotationAndShear (mat, result0) {
		var position = [], scale = [], aShear = [], aRotation = [];
		mat3.decomposeAffine(mat, position, scale, aShear, aRotation);

		// FIXME: DRY this copy from affine which i used to avoid computing the angle
		// but then also adapted for shearX = 0
		var px = position[0], py = position[1],
			scaleX = scale[0], scaleY = scale[1];

		var result = result0 || mat3();
		
		result[0] = scaleX;
		result[1] = 0;
		result[2] = 0;
		result[3] = 0;
		result[4] = scaleY;
		result[5] = 0;
		result[6] = px;
		result[7] = py;
		result[8] = 1;
	}

	function calcAttachmentCoord (forSubLayer) {
		/*jshint validthis: true */
		var	parentLayer = forSubLayer.parentPuppet.getParentLayer(),
			// note: tag coordinates are expressed in the source frame of the parent layer
			attachment = this.convertPointFrom(forSubLayer.getTag(), parentLayer),
			points = [ attachment ],
			warper = this.getWarperContainer();

		// TODO: aggregate and optimize into one coord computation
		var coords = warper.calcPointCoord(points);
		coords.inPoints = points;

		return coords;
	}

	function Layer(source, args0) {
		this.StageId("Layer_");
		this.source = source;

		// patch motionMode setting
		// TODO: remove patch in preview 3
		if (args0.motionMode === false) {
			delete args0.motionMode;
		}

		lodash.defaults(this, args0, {
			name : null,
			bindingId : null,
			tag : [0, 0],
			tagBackstageId : null,
			matLayer_Source : mat3.identity(),
			motionMode : { translation : true, linear : true },
			warpWithParent : true,
			bUserVisible : true, // actually more like SDK-visible on the JS side (we use this to hide Locations layers)
			bTriggerable : false,	// if true, will also have a bHideSiblingsWhenTriggered
			warpHash : null,
			keyTrigger0 : null,
			triggeredBy : null,
			isAttached : true,
			bindingOpacity : 1,
			bindingBlendMode: null,
			bIsClipped : false,
			layerTags: {},
		});

		/**
		 * Motion State
		 * The top-level layer will store state for itself and all descendents.
		 */
		this.backingTomNow = false; // lazy init on first get
		this.backingTomNext = false;
		this.tomPrev = false;

		// translate motionMode setting to automation object
		lodash.forEach({"position" : "translation", "scaling" : "linear"}, function (newKey, oldKey) {
			if (lodash.has(this.motionMode, oldKey)) {
				this.motionMode[newKey] = this.motionMode[oldKey];
				delete this.motionMode[oldKey];
			}
		}.bind(this));		

		this.handleTreeArray = null;
		this.handleJoint = null;
		this.matSource_Layer = mat3.invert(this.matLayer_Source);
		this.keyTrigger0 = (args0 && args0.keyTrigger0) || null;
		this.triggeredBy = (args0 && args0.triggeredBy) || null;

		// assign memoized functions
		this.getAttachmentCoord = lodash.memoize(calcAttachmentCoord, StageId.hash);
		
		if (this.keyTrigger0) {
			this.keyTrigger0._triggerKey = this.keyTrigger0._triggerKey.toUpperCase();
		}

		// parentPuppet is set by the parent puppet when the layer is installed in the puppet
		this.parentPuppet = null;
		this.sdkLayer = new SdkLayer(this);

		// create display item containers for this layer:
		// tag -> matrix -> source
		this.DisplayContainerStorage(false, false, this.name);

		var cLayerAttacher = new Container("__LayerAttacher__" + this.name);
		cLayerAttacher.setMatrix(mat3.translation(this.tag));

		var cSourceAttacher = new Container("__SourceAttacher__" + this.name);
		cLayerAttacher.addChild(cSourceAttacher, 0);
		cSourceAttacher.setMatrix(this.matLayer_Source);

		this.source.parentLayer = this; // HACK, so that layer params can resolve to the root layer (one up from the root puppet)
									// and so we can go from stagePuppet to stageLayer (similar to parentPuppet for going up from layer)

		var puppet = this.getPuppet();									
		if (puppet) {
			var cPuppet = puppet.getDisplayContainer();
			cSourceAttacher.addChild(cPuppet);
			this.handleJoint = setupJoint(this);
		} else {
			// or use the source itself as the display item
			cSourceAttacher.addChild(this.getSource());
		}

		this.setDisplayContainer(cLayerAttacher);
	}

	// export to keep in synch
	Layer.kOriginLeaf = kOriginLeaf;

	utils.mixin(Layer, DisplayContainerStorage, StageId, {

		getSdkLayer : function () { return this.sdkLayer; },
		
		getName : function () { return this.name; },	// note: matching container also has this name
		
		getSource : function () { return this.source; },

		getSourceMatrixRelativeToLayer : function () {
			return mat3.clone(this.matLayer_Source);
		},

		getLayerMatrixRelativeToSource : function () {
			return mat3.clone(this.matSource_Layer);
		},

		getPuppet : function () {
			var p = this.getSource();
			return p.constructor === Puppet ? p : null;
		},

		getTag : function () {
			return vec2.from(this.tag);			
		},

		getTagBackstageId : function () {
			var id = this.tagBackstageId;
			return id ? id.slice(0) : null;
		},
		
		getBindingId: function () {
			return this.bindingId;
		},

		// may return null
		getKeyTriggerData: function () {
			return this.keyTrigger0;
		},

		getKeyTriggerKey : function () {
			var triggerData0 = this.getKeyTriggerData();
			return (triggerData0 && triggerData0._triggerKey) || null; 
		},

		getHideOthersWhenTriggered : function () { 
			var triggerData0 = this.getKeyTriggerData();
			return (triggerData0 && triggerData0._bHideOthersWhenTriggered) || false; 
		},

		getTriggerUsesLatch : function () { 
			var triggerData0 = this.getKeyTriggerData();
			return (triggerData0 && (triggerData0._triggerLatch === 2)) || false; 
		},
		
		setVisible : function (bEnabled) {
			var c = this.getDisplayContainer();
			c.setVisibleEnabled(bEnabled);
		},

		getVisible : function () { 
			var c = this.getDisplayContainer();
			return c.getVisibleEnabled();
		},
		
		setTriggerable :	function (bHideSiblings0)	{	
														 	this.bTriggerable = true;
														 	this.bHideSiblingsWhenTriggered = !!bHideSiblings0;
														},

		getTriggerable :	function ()					{ return this.bTriggerable; },
		getHideSiblingsWhenTriggered : function ()		{ return this.bHideSiblingsWhenTriggered; },
		
		clearPerFrameTriggerData : function ()			{
															this.bTriggered = false;
//															if (this.triggerMutexPriority) {
//																console.logToUser("clearing triggerMutexPriority for " + this.getName());
//															}
															delete this.triggerMutexPriority;
														},

		setTriggered :		function (bTriggered, priority0) {		// see also clearPerFrameTriggerData
			utils.assert(this.bTriggerable, "attempt to trigger untriggerable");
            
            //console.logToUser(`triggering ${this.getName()} with pri ${priority0}`);
			
			var priority = priority0 || 0.0,
				parentSdkLayer = this.getSdkLayer().getParentLayer(),	// store priority in parent, since it's common to all the children
				parentLayer = parentSdkLayer.privateLayer,
				prevMutexPriority = parentLayer.triggerMutexPriority,
				bPriorPri = (prevMutexPriority !== undefined),	// has it ever been set before
				bMutex = this.bHideSiblingsWhenTriggered;
			
			if (bMutex && bPriorPri && priority > prevMutexPriority) {
				// clear out previous triggers when a higher priority comes by; for common case of no priority specified,
				//	all the triggers land, and the one closest to the front layer wins during mediation
				parentSdkLayer.forEachDirectChildLayer(function (lay) {
					lay.privateLayer.bTriggered = false;
				});
			}
			
			if (bMutex) {
				if (!bPriorPri || priority >= prevMutexPriority) {
					this.bTriggered = bTriggered;

					//console.logToUser(`setting triggerMutexPriority = ${priority} for ${this.getName()}`);
					parentLayer.triggerMutexPriority = priority;
				}
			} else {
				this.bTriggered = bTriggered;
			}
		},
		
		getTriggered :		function ()					{ return this.bTriggered; },
		getTriggeredByMap :	function ()					{ return this.triggeredBy; },
		getLayerTags:		function ()					{ return this.layerTags; },
		
		setWarpWithParent : function (bEnabled, aMatPuppet_HandleAtRest0) {
			var cSourceAttacher, cWarper, cPuppet,
				didWarpWithParent = this.getWarpWithParent(),
				puppet = this.getPuppet();

			this.warpWithParent = bEnabled;

			if ( !puppet ) utils.assert(this.warpWithParent, "NYI: skin layers that warp on their own.");

			// no change
			if ( !puppet || (didWarpWithParent === this.warpWithParent) ) return;

			// layer no longer warps on its own
			if ( !didWarpWithParent && this.warpWithParent) {
				this.handleJoint = setupJoint(this);
				// remove warper
				cSourceAttacher = this.getSourceAttachContainer();
				cPuppet = puppet.getDisplayContainer();
				cSourceAttacher.removeChildAtIndex(0);
				cSourceAttacher.addChild(cPuppet, 0);
				return;
			}

			// layer starts to warp on its own
			if ( didWarpWithParent && !this.warpWithParent ) {
				this.handleJoint =setupJoint(this);
				// make warper
				cSourceAttacher = this.getSourceAttachContainer();
				cWarper = new Container("__LayerWarper__" + this.name);
				cPuppet = puppet.getDisplayContainer();
				cSourceAttacher.removeChildAtIndex(0);
				cSourceAttacher.addChild(cWarper);

				cWarper.setWarpDomain(puppet.getWarpType());
				cWarper.setWarpHash(this.warpHash);
				cWarper.setWarpMeshExpansion(puppet.getWarpMeshExpansion());
				cWarper.setWarpMaxNumTrianglesHint(puppet.getWarpMaxNumTrianglesHint());
				cWarper.setWarpMinTriangleAreaHint(puppet.getWarpMinTriangleAreaHint());

				cWarper.setParentCanWarpMe(this.warpWithParent);
				cWarper.addChild(cPuppet);

				var tree = this.getHandleTreeArray(),
					aLeafRef = tree.getLeafRefArray(),
					aMatPuppet_HandleSource = tree.getSourceFrames("puppet");

				// handle dof description
				if (aMatPuppet_HandleAtRest0) {
					this.aMatPuppet_HandleAtRest = aMatPuppet_HandleAtRest0;
				} else {
					this.aMatPuppet_HandleAtRest = tree.getAccumulatedTreeAtRest("puppet");
				}
				var aMatPuppet_LeafAtRest = lodash.at(this.aMatPuppet_HandleAtRest, aLeafRef);

				// handle shape description
				var aLeaf = lodash.at(tree.aHandle, aLeafRef),
					aMatPuppet_LeafPuppet = lodash.at(aMatPuppet_HandleSource, aLeafRef),

					aLocations = aLeaf.map(function (li, i) {
						var mi = aMatPuppet_LeafPuppet[i];
						return li.transformLocations(mi);
					}),

					aAnchors = aLeaf.map(function (li, i) {
						var mi = aMatPuppet_LeafPuppet[i];
						return li.transformAnchor(mi);
					});


				cWarper.setWarpRest(aMatPuppet_LeafAtRest, aLocations, aAnchors);
			}

		},

		getWarpWithParent : function () { 
			return this.warpWithParent;
		},
		
		getUserVisible : function () {
			return this.bUserVisible;
		},

		clone: function (result) {
			var source = this.source.clone(),
				args = {
					name : this.name ? this.name.slice(0) : null,
					tag : this.getTag(),
					tagBackstageId : this.getTagBackstageId(),
					matLayer_Source : mat3.clone(this.matLayer_Source),
					motionMode : lodash.clone(this.motionMode),
					bindingId : this.bindingId,
					bUserVisible : this.bUserVisible,
					keyTrigger0 : this.keyTrigger0,
					triggeredBy : this.triggeredBy,
					isAttached : this.isAttached,
					bIsClipped : this.bIsClipped,
					layerTags: lodash.clone(this.layerTags),
				};

			if (result) {
				Layer.call(result, source, args);
			} else {
				result = new Layer(source, args);
			}
			
			var bOrigVisible = this.getVisible();
			
			result.bInitialVisibility	= bOrigVisible; // retain current state (mediator uses this on every frame for untriggered things)
			result.bTriggerable 		= false; // don't adopt this.bTriggerable or bHideSiblingsWhenTriggered since behaviors
												 // aren't running on cloned particles, nothing gets triggered
			// (and in the future, if they do, they should get an onCreateStageBehavior call where triggerability would get set again)

			// clone display container properties
			result.setOpacity(this.getOpacity());
			result.setBlendMode(this.getBlendMode());
			result.setVisible(bOrigVisible);
			result.setWarpWithParent(this.getWarpWithParent(), lodash.cloneDeep(this.aMatPuppet_HandleAtRest));

			return result;
		},

		getSourceAttachContainer : function () {
			var cLayerAttacher = this.getDisplayContainer();
			return cLayerAttacher.getChildren()[0];
		},

		getWarperContainer : function () {
			if (this.getWarpWithParent()) return null;

			var cSourceAttacher = this.getSourceAttachContainer(),
				cWarper = cSourceAttacher.getChildren()[0];

			utils.assert(cWarper, "getWarperContainer(): warper container not found.");
			return cWarper;
		},

		displayInView : function (canGenerateMesh) {
			// initialize warp
			var cWarper = this.getWarperContainer();
			if (cWarper) {
				// FIXME: This should not be hard coded
				var needsMesh = !lodash.isUndefined(this.getLayerTags()["Adobe.Physics.Collide"]) ||
								!lodash.isUndefined(this.getLayerTags()["Adobe.Physics.Throw"]);

				if ( needsMesh && canGenerateMesh ) {
					cWarper.initGeometryComponent();
				}

				cWarper.initWarp();
				if ( cWarper.shouldInitializeWarp() && !cWarper.canWarp() ) {
					console.logToUser("A layer binding puppet '" + this.getPuppet().getName() + "' has several handles but no artwork to warp.");
				}
			}
			// and propagate display event
			var puppet = this.getPuppet();
			if (puppet) {
				puppet.displayInView(canGenerateMesh);
			}
		},

		getTrackItem : function () {
			// HACK we should not go to source puppet for this.
			// TODO revise when we stamp out reliance on Puppet instead of Layer.
			var puppet = this.getPuppet();
			utils.assert(puppet, "NYI: getView on skin layers.");
			return puppet.getTrackItem();
		},

		getView : function () {
			return this.getTrackItem().getView();
		},

		getTrack : function () {
			var ti = this.getTrackItem();
			if (ti) {
				return ti.getParent();
			}
			return null;
		},

		getScene : function () {
			var t = this.getTrack();
			if (t) {
				return t.getParent();
			}
			return null;
		},

		setOpacity : function (opacity)	{	
			var c = this.getDisplayContainer();
			utils.assert(opacity+1e-6 >= 0 && opacity-1e-6 <= 1.0, "opacity must be between 0 and 1"); // Add some tolerance
			if (opacity < 0) {
				opacity = 0;
			} else if (opacity > 1) {
				opacity = 1;
			}
		 	c.setAlpha(opacity);
		},

		getBindingOpacity : function () {	
			return this.bindingOpacity;
		},

		getOpacity : function ()	{	
			var c = this.getDisplayContainer();
		 	return c.getAlpha();
		},

		setBlendMode : function (blendMode)	{	
			var c = this.getDisplayContainer();
		 	return c.setBlendMode(blendMode);
		},

		getBindingBlendMode : function () {	
			return this.bindingBlendMode;
		},

		getBindingIsClipped : function () {
			return this.bIsClipped;
		},
		
		getBlendMode : function ()	{	
			var c = this.getDisplayContainer();
		 	return c.getBlendMode();
		},

		getHandleTreeRoot : function () {
			var puppet = this.getPuppet();
			utils.assert(puppet, "getHandleTreeRoot(): puppet not found.");
			return puppet.getHandleTreeRoot();
		},

		getHandleJoint : function () {
			return this.handleJoint;
		},

		filterJointFrame : function (mat, result0) {
			var result = result0 || mat3(),
				mode = this.motionMode;

			if (mode.translation === false && mode.linear === true) {

				// hinge propagates position and scale
				matRemoveRotationAndShear(mat, result);

			} else if (mode.translation === false && mode.linear === false) {

				// weld propagate everything but shear
				matRemoveShear(mat, result);

			} else {

				// free propagate everything but shear
				matRemoveShear(mat, result);

			}

			// FIXME: create another type that will propagate skew for regular hierarchical transform behavior

			return result;
		},

		/**
		 * Gather leaf handles that warp together.
		 * Co-recursive with the corresponding Puppet function.
		 * @return Pre-order array of handles.
		 */
		gatherHandleLeafArray : function () {
			var tree = this.getHandleTreeArray(),
				aLeafRef = tree.getLeafRefArray();

			return lodash.at(tree.aHandle, aLeafRef);
		},

		getHandleMatrixAtRest : function (handle, result0) {
			var tree = this.getHandleTreeArray(),
				handleRef = tree.getHandleRef(handle);
			utils.assert(handleRef !== null, "getHandleMatrixAtRest(): handle not found.");

			return tree.getAccumulatedHandleAtRest("layer", handleRef, result0);
		},

		// HACK: refactor along with parentLayer and parentPuppet fields.
		getWarperLayer : function () {
			var warperLayer = this;
			// note: parentPuppet not defined for layer-like track item
			while ( warperLayer && warperLayer.getWarpWithParent() && warperLayer.parentPuppet) {
				warperLayer = warperLayer.parentPuppet.getWarperLayer();
			}

			// FIXME: when it's appropriate to investigate more disruptive changes uncomment the following assert and chase down whenever it fires...
			// utils.assert(!warperLayer.getWarpWithParent(), "getWarperLayer(): warper layer not warping independently.");
			utils.assert(warperLayer, "getWarperLayer(): warper layer not found.");
			return warperLayer;
		},

		/**
		 * Return warper frame as a matrix relative to the frame of its source item.
		 * The warper frame is defined by the initial configuration of the warping layer. 
		 */
		warperFrame : function (matSource_Warper0) {
			var candidate = this,
				matSource_Candidate = matSource_Warper0 || mat3(),
				matCandidate_Parent = mat3();

			mat3.identity(matSource_Candidate);

			// note: parentPuppet not defined for layer-like track item
			while ( candidate && candidate.getWarpWithParent() && candidate.parentPuppet) {
				mat3.multiply(candidate.matSource_Layer, mat3.translation(vec2.negate(candidate.tag)), matCandidate_Parent);

				candidate = candidate.parentPuppet.getParentLayer();
				mat3.multiply(matSource_Candidate, matCandidate_Parent, matSource_Candidate);
			}
			
			utils.assert(candidate, "getWarperLayer(): warper layer not found.");
			return matSource_Candidate;
		},

		/**
		 * @private
		 */		
		getHandleTreeArray : function () {
			if (this.handleTreeArray !== null) return this.handleTreeArray;

			var p0 = this.getPuppet();
			this.handleTreeArray = p0 ? p0.privateGatherHandleTreeArray() : null;
			return this.handleTreeArray;
		},

		// Provided in constructor as memoized function for this layer
		getAttachmentCoord : false,

		getWarpAtAttachment : function (withDofs, forLayer) {
			var coords = this.getAttachmentCoord(forLayer),
				delta = vec2.subtract(coords.inPoints[0], coords.points[0]);

			var [ matWarper_Tag ] = this.getWarperContainer().warpPoint(withDofs, coords);

			// modify transform when there is a difference between original tag and its mesh projection
			vec2.transformAffine(matWarper_Tag, delta, delta);
			mat3.setTranslation(delta, matWarper_Tag);

			return matWarper_Tag;
		},

		/**
		 * Convert point coordinates from the source of another layer to the source of this layer.
		 */
		// FIXME: move into Puppet as it works with source aka puppet and create another for Layer.
		convertPointFrom : function (point, fromLayer) {
			var toLayer = this,
				result = vec2.from(point),
				matParent_Child = mat3();

			while ( fromLayer && fromLayer.parentPuppet && fromLayer !== toLayer ) {
				// matParent_Child = matParent_Layer matLayer_Source, note tag is the origin of Layer
				mat3.multiply(mat3.translation(fromLayer.tag), fromLayer.matLayer_Source, matParent_Child);
				vec2.transformAffine(matParent_Child, result, result);
				fromLayer = fromLayer.parentPuppet.getParentLayer();				
			}

			utils.assert(fromLayer === toLayer, "convertPointFrom(): can only convert to the source of an ancestor layer.");

			return result;
		},


		// Degrees of Freedom
		prepareState (shouldReset0) {
			var shouldReset = lodash.isUndefined(shouldReset0) ? false : shouldReset0;
			dev.MatrixDofs.doPreMediateLayer(this, shouldReset);
			dev.TransformDofs.doPreMediateLayer(this, shouldReset);
		}, 

		commitState () {
			ZoneProfiler.push("applyTransformDof");
			dev.TransformDofs.doPostMediateLayer(this, "tLayerTaskItems");
			ZoneProfiler.pop("applyTransformDof");

			ZoneProfiler.push("applyMatrixDof");
			dev.MatrixDofs.doPostMediateLayer(this);
			ZoneProfiler.pop("applyMatrixDof");
		},

		// Special purpose commit for use by dangle or rigid-body simulations.
		// Unlike the default commitState it commits the state in Matrix Dofs only,
		// AND does not return unaltered handles back to rest.
		commitMatrixDofs : config.MutableTree.enabled ?
            function () {
                treeOfMatrix.warp(this, this.tomPrevPrev, this.tomPrev, this.tomNow);

                this.tomPrevPrev = this.tomPrev;
                this.tomPrev = this.tomNow.clone();       // TODO: factor with doPostMediateLayer
            }
        :
            function () {

                var tomPrev0 = this.tomPrev,
                        tomNow = this.tomNow,
                        tomNext = this.tomNext;

                // log("==========================================================================");
                // log(`SCENE layer '${this.getName()}', puppet '${this.getPuppet().getName()}'\n\tnow\n${tomNow}\n\tnext\n${tomNext}`);
                // dev.MatrixDofs.logTreeDiffs(tomNow, tomNext);

                var tomUpdated = dofs.Tree.updateValuesWithZipped(dev.MatrixDofs.updateOnMatrixChange, tomNow, tomNext);

                // log(`NEXT\n${tomUpdated}`);

                var tomWarp = treeOfMatrix.warp(this, tomPrev0, tomNow, tomUpdated);

                if (tomWarp !== tomNow) {
                    // log(`tomWarp\n${tomWarp}`);
                    dev.MatrixDofs.logTreeDiffs(tomNow, tomWarp);
                }			

                this.tomPrev = tomNow;
                this.tomNow = tomWarp;
                this.tomNext = tomWarp;
            },

		isScenePuppetLayer () {
			const puppet0 = this.getPuppet();
			if (puppet0) {
				// FIXME: Use bScenePuppet from Lua or keep in sink with the assigned name for *scene* puppet.
				return lodash.startsWith(puppet0.getName(), "Scene Puppet");
			}
		},
	});

    /*
    function callsiteInfo (fn) {
        var e = new Error();
        Error.captureStackTrace(e, fn);
        return e.stack.substring(6);
    }

    function traceCall (name) {
        var fn = function () {
            console.logToUser(`${name}\n${callsiteInfo(fn)}`);
        };
        fn();
    }*/

    if (config.MutableTree.enabled) {
        // Temporary setter/getter for lazy init of layer state.
        Object.defineProperty(Layer.prototype, "tomNow", { 
            get: function () { 
                var tomNow = this.backingTomNow;
                if (tomNow === false) {
                    tomNow =  this.tomBirth.clone();
                    this.backingTomNow = tomNow;
                }

                return tomNow;
            },

            set: function (val) {
                this.backingTomNow = val;
            }
        });

        Object.defineProperty(Layer.prototype, "tomBirth", {
            get: function () { 
                var backing = this.backingTomBirth;
                if (!backing) {
                    backing = treeOfMatrix.init(this);
                    this.backingTomBirth = backing;
                }

                return backing;
            }
        });
        
        Object.defineProperty(Layer.prototype, "tomPrev", {
            get: function () { 
                var tomPrev = this.backingTomPrev;
                if (tomPrev === false) {
                    tomPrev =  this.tomBirth;    // no need to clone because both are read-only
                    this.backingTomPrev = tomPrev;
                }

                return tomPrev;
            },

            set: function (val) {
                this.backingTomPrev = val;
            }
        });
    } else {
        // Temporary setter/getter for lazy init of layer state.
        Object.defineProperty(Layer.prototype, "tomNow", { 
            get: function () { 
                var tomNow = this.backingTomNow;
                if (tomNow === false) {
                    tomNow = this.tomBirth;
                    this.backingTomNow = tomNow;
                }

                return tomNow;
            },

            set: function (val) {
                this.backingTomNow = val;
            }
        });

        Object.defineProperty(Layer.prototype, "tomNext", { 
            get: function () { 
                return this.backingTomNext;
            },

            set: function (val) {

                // if (this.backingTomNext === val) {
                // 	traceCall("SET tomNext");
                // 	console.logToUser("SAME");
                // } else {
                // 	traceCall("SET tomNext");
                // 	console.logToUser(`tomNext\n${val}`);
                // }
                this.backingTomNext = val;
            }
        });

        Object.defineProperty(Layer.prototype, "tomBirth", { 
            get: function () { 
                var backing = this.backingTomBirth;
                if (!backing) {
                    backing = treeOfMatrix.init(this);
                    this.backingTomBirth = backing;
                }

                return backing;
            }
        });
    }

	return Layer;
});
